/*
 * Decompiled with CFR 0.152.
 */
package gov.sandia.cognition.math;

import gov.sandia.cognition.annotation.PublicationReference;
import gov.sandia.cognition.annotation.PublicationType;
import gov.sandia.cognition.collection.NumberComparator;
import gov.sandia.cognition.math.MathUtil;
import gov.sandia.cognition.math.ProbabilityUtil;
import gov.sandia.cognition.util.DefaultPair;
import gov.sandia.cognition.util.Pair;
import gov.sandia.cognition.util.WeightedValue;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;

public class UnivariateStatisticsUtil {
    @PublicationReference(title="Algorithms for calculating variance", type=PublicationType.WebPage, year=2010, author={"Wikipedia"}, url="http://en.wikipedia.org/wiki/Algorithms_for_calculating_variance")
    public static double computeMean(Iterable<? extends Number> data) {
        int n = 0;
        double mean = 0.0;
        for (Number number : data) {
            double x = number.doubleValue();
            double delta = x - mean;
            mean += delta / (double)(++n);
        }
        return mean;
    }

    @PublicationReference(title="Algorithms for calculating variance", type=PublicationType.WebPage, year=2010, author={"Wikipedia"}, url="http://en.wikipedia.org/wiki/Algorithms_for_calculating_variance")
    public static double computeWeightedMean(Iterable<? extends WeightedValue<? extends Number>> data) {
        double mean = 0.0;
        double weightSum = 0.0;
        for (WeightedValue<? extends Number> weightedValue : data) {
            double x = weightedValue.getValue().doubleValue();
            double weight = weightedValue.getWeight();
            if (weight == 0.0) continue;
            if (weight < 0.0) {
                weight = -weight;
            }
            double newWeightSum = weightSum + weight;
            double delta = x - mean;
            double update = delta * weight / newWeightSum;
            mean += update;
            weightSum = newWeightSum;
        }
        return mean;
    }

    public static double computeVariance(Collection<? extends Number> data) {
        return UnivariateStatisticsUtil.computeMeanAndVariance(data).getSecond();
    }

    public static double computeVariance(Collection<? extends Number> data, double mean) {
        int num = data.size();
        if (num < 2) {
            return 0.0;
        }
        double biasedVariance = UnivariateStatisticsUtil.computeCentralMoment(data, mean, 2);
        double unbiasedVariance = (double)num / ((double)num - 1.0) * biasedVariance;
        return unbiasedVariance;
    }

    public static double computeRootMeanSquaredError(Collection<? extends Number> data) {
        double mean = UnivariateStatisticsUtil.computeMean(data);
        return UnivariateStatisticsUtil.computeRootMeanSquaredError(data, mean);
    }

    public static double computeRootMeanSquaredError(Collection<? extends Number> data, double mean) {
        double sse = UnivariateStatisticsUtil.computeSumSquaredDifference(data, mean);
        double rms = data.size() > 0 ? Math.sqrt(sse / (double)data.size()) : 0.0;
        return rms;
    }

    public static double computeSum(Iterable<? extends Number> data) {
        double sum = 0.0;
        for (Number number : data) {
            sum += number.doubleValue();
        }
        return sum;
    }

    public static double computeSumSquaredDifference(Iterable<? extends Number> data, double target) {
        double sum = 0.0;
        for (Number number : data) {
            double delta = number.doubleValue() - target;
            sum += delta * delta;
        }
        return sum;
    }

    @PublicationReference(author={"Wikipedia"}, title="Pearson product-moment correlation coefficient", type=PublicationType.WebPage, year=2011, url="http://en.wikipedia.org/wiki/Pearson_product-moment_correlation_coefficient")
    public static double computeCorrelation(Collection<? extends Number> data1, Collection<? extends Number> data2) {
        double b2;
        if (data1.size() != data2.size()) {
            throw new IllegalArgumentException("Datasets must be the same size!");
        }
        int num = data1.size();
        if (num <= 0) {
            return 1.0;
        }
        Iterator<? extends Number> i1 = data1.iterator();
        Iterator<? extends Number> i2 = data2.iterator();
        double sum1 = 0.0;
        double sum2 = 0.0;
        double sum11 = 0.0;
        double sum22 = 0.0;
        double sum12 = 0.0;
        for (int n = 0; n < num; ++n) {
            double value1 = i1.next().doubleValue();
            double value2 = i2.next().doubleValue();
            sum1 += value1;
            sum2 += value2;
            sum11 += value1 * value1;
            sum22 += value2 * value2;
            sum12 += value1 * value2;
        }
        double top = (double)num * (sum12 - sum1 / (double)num * sum2);
        double b1 = Math.sqrt((double)num * (sum11 - sum1 / (double)num * sum1));
        double bottom = b1 * (b2 = Math.sqrt((double)num * (sum22 - sum2 / (double)num * sum2)));
        double retval = bottom <= 0.0 ? 1.0 : top / bottom;
        return retval;
    }

    public static double computeMedian(Collection<? extends Number> data) {
        return UnivariateStatisticsUtil.computePercentile(data, 0.5);
    }

    public static double computePercentile(Collection<? extends Number> data, double percentile) {
        ProbabilityUtil.assertIsProbability(percentile);
        ArrayList<? extends Number> sortedData = new ArrayList<Number>(data);
        Collections.sort(sortedData, new NumberComparator());
        int num = data.size();
        double numPct = (double)(num - 1) * percentile;
        double remainder = Math.abs(Math.IEEEremainder(numPct, 1.0));
        int lowerIndex = (int)Math.floor(numPct);
        double lower = sortedData.get(lowerIndex).doubleValue();
        if (remainder == 0.0) {
            return lower;
        }
        int upperIndex = lowerIndex + 1;
        double upper = sortedData.get(upperIndex).doubleValue();
        return lower * (1.0 - remainder) + upper * remainder;
    }

    public static double computeMinimum(Iterable<? extends Number> data) {
        double minimum = Double.MAX_VALUE;
        for (Number number : data) {
            double x = number.doubleValue();
            if (!(minimum > x)) continue;
            minimum = x;
        }
        return minimum;
    }

    public static double computeMaximum(Iterable<? extends Number> data) {
        double maximum = -1.7976931348623157E308;
        for (Number number : data) {
            double x = number.doubleValue();
            if (!(maximum < x)) continue;
            maximum = x;
        }
        return maximum;
    }

    public static Pair<Double, Double> computeMinAndMax(Iterable<? extends Number> data) {
        double min = Double.POSITIVE_INFINITY;
        double max = Double.NEGATIVE_INFINITY;
        for (Number number : data) {
            double x = number.doubleValue();
            if (min > x) {
                min = x;
            }
            if (!(max < x)) continue;
            max = x;
        }
        return DefaultPair.create(min, max);
    }

    @PublicationReference(author={"Wikipedia"}, title="Skewness", type=PublicationType.WebPage, year=2009, url="http://en.wikipedia.org/wiki/Skewness")
    public static double computeSkewness(Collection<? extends Number> data) {
        int num = data.size();
        if (num < 3) {
            return 0.0;
        }
        Pair<Double, Double> pair = UnivariateStatisticsUtil.computeMeanAndVariance(data);
        double mean = pair.getFirst();
        double stddev = Math.sqrt(pair.getSecond());
        double moment3 = UnivariateStatisticsUtil.computeCentralMoment(data, mean, 3);
        double biasedSkewness = moment3 / Math.pow(stddev, 3.0);
        return biasedSkewness;
    }

    public static double computeCentralMoment(Iterable<? extends Number> data, double mean, int moment) {
        if (moment < 1) {
            throw new IllegalArgumentException("Moment must be >= 1");
        }
        int num = 0;
        double sum = 0.0;
        for (Number number : data) {
            double delta = number.doubleValue() - mean;
            sum += Math.pow(delta, moment);
            ++num;
        }
        if (num < 1) {
            return 0.0;
        }
        return sum / (double)num;
    }

    public static double computeWeightedCentralMoment(Iterable<? extends WeightedValue<? extends Number>> data, double mean, int moment) {
        if (moment < 1) {
            throw new IllegalArgumentException("Moment must be >= 1");
        }
        int num = 0;
        double sum = 0.0;
        double weightSum = 0.0;
        for (WeightedValue<? extends Number> weightedValue : data) {
            double weight = Math.abs(weightedValue.getWeight());
            if (weight != 0.0) {
                double delta = weightedValue.getValue().doubleValue() - mean;
                sum += weight * Math.pow(delta, moment);
                weightSum += weight;
            }
            ++num;
        }
        if (num < 1) {
            return 0.0;
        }
        return sum / weightSum;
    }

    @PublicationReference(author={"Wikipedia"}, title="Kurtosis", type=PublicationType.WebPage, year=2009, url="http://en.wikipedia.org/wiki/Kurtosis")
    public static double computeKurtosis(Collection<? extends Number> data) {
        if (data.size() < 2) {
            return 0.0;
        }
        Pair<Double, Double> pair = UnivariateStatisticsUtil.computeMeanAndVariance(data);
        double mean = pair.getFirst();
        double variance = pair.getSecond();
        double moment4 = UnivariateStatisticsUtil.computeCentralMoment(data, mean, 4);
        double biasedKurtosis = moment4 / (variance * variance) - 3.0;
        return biasedKurtosis;
    }

    @PublicationReference(author={"Wikipedia"}, title="Kurtosis", type=PublicationType.WebPage, year=2009, url="http://en.wikipedia.org/wiki/Kurtosis")
    public static double computeWeightedKurtosis(Collection<? extends WeightedValue<? extends Number>> data) {
        if (data.size() < 2) {
            return 0.0;
        }
        Pair<Double, Double> pair = UnivariateStatisticsUtil.computeWeightedMeanAndVariance(data);
        double mean = pair.getFirst();
        double variance = pair.getSecond();
        double moment4 = UnivariateStatisticsUtil.computeWeightedCentralMoment(data, mean, 4);
        double biasedKurtosis = moment4 / (variance * variance) - 3.0;
        return biasedKurtosis;
    }

    @PublicationReference(author={"Wikipedia"}, title="Entropy (information theory)", type=PublicationType.WebPage, year=2009, url="http://en.wikipedia.org/wiki/Entropy_(Information_theory)")
    public static double computeEntropy(Iterable<? extends Number> data) {
        double entropy = 0.0;
        for (Number number : data) {
            double p = number.doubleValue();
            if (p == 0.0) continue;
            entropy -= p * MathUtil.log2(p);
        }
        return entropy;
    }

    @PublicationReference(title="Algorithms for calculating variance", type=PublicationType.WebPage, year=2010, author={"Wikipedia"}, url="http://en.wikipedia.org/wiki/Algorithms_for_calculating_variance")
    public static Pair<Double, Double> computeMeanAndVariance(Iterable<? extends Number> data) {
        int n = 0;
        double mean = 0.0;
        double m2 = 0.0;
        for (Number number : data) {
            double x = number.doubleValue();
            double delta = x - mean;
            m2 += delta * (x - (mean += delta / (double)(++n)));
        }
        double variance = n >= 2 ? m2 / (double)(n - 1) : 0.0;
        return DefaultPair.create(mean, variance);
    }

    @PublicationReference(title="Algorithms for calculating variance", type=PublicationType.WebPage, year=2010, author={"Wikipedia"}, url="http://en.wikipedia.org/wiki/Algorithms_for_calculating_variance")
    public static Pair<Double, Double> computeWeightedMeanAndVariance(Iterable<? extends WeightedValue<? extends Number>> data) {
        double mean = 0.0;
        double weightSum = 0.0;
        double m2 = 0.0;
        for (WeightedValue<? extends Number> weightedValue : data) {
            double x = weightedValue.getValue().doubleValue();
            double weight = weightedValue.getWeight();
            if (weight == 0.0) continue;
            if (weight < 0.0) {
                weight = -weight;
            }
            double newWeightSum = weightSum + weight;
            double delta = x - mean;
            double update = delta * weight / newWeightSum;
            m2 += weightSum * delta * update;
            mean += update;
            weightSum = newWeightSum;
        }
        double variance = weightSum > 0.0 ? m2 / weightSum : 0.0;
        return DefaultPair.create(mean, variance);
    }
}

