/*
 * Decompiled with CFR 0.152.
 */
package org.openimaj.math.geometry.transforms;

import Jama.Matrix;
import Jama.SingularValueDecomposition;
import java.util.ArrayList;
import java.util.List;
import org.openimaj.citation.annotation.Reference;
import org.openimaj.citation.annotation.ReferenceType;
import org.openimaj.math.geometry.point.Coordinate;
import org.openimaj.math.geometry.point.Point2d;
import org.openimaj.math.geometry.point.Point2dImpl;
import org.openimaj.math.geometry.shape.Rectangle;
import org.openimaj.math.matrix.MatrixUtils;
import org.openimaj.util.pair.IndependentPair;
import org.openimaj.util.pair.Pair;

public class TransformUtilities {
    private TransformUtilities() {
    }

    public static Matrix rotationMatrix(double rot) {
        Matrix matrix = Matrix.constructWithCopy(new double[][]{{Math.cos(rot), -Math.sin(rot), 0.0}, {Math.sin(rot), Math.cos(rot), 0.0}, {0.0, 0.0, 1.0}});
        return matrix;
    }

    public static Matrix translateToPointMatrix(Point2d from, Point2d to) {
        Matrix matrix = Matrix.constructWithCopy(new double[][]{{1.0, 0.0, to.minus(from).getX()}, {0.0, 1.0, to.minus(from).getY()}, {0.0, 0.0, 1.0}});
        return matrix;
    }

    public static Matrix translateMatrix(double x, double y) {
        Matrix matrix = Matrix.constructWithCopy(new double[][]{{1.0, 0.0, x}, {0.0, 1.0, y}, {0.0, 0.0, 1.0}});
        return matrix;
    }

    public static Matrix centeredRotationMatrix(double rot, int width, int height) {
        int halfWidth = Math.round(width / 2);
        int halfHeight = Math.round(height / 2);
        return TransformUtilities.rotationMatrixAboutPoint(rot, halfWidth, halfHeight);
    }

    public static Matrix scaleMatrixAboutPoint(double sx, double sy, int tx, int ty) {
        return Matrix.identity(3, 3).times(TransformUtilities.translateMatrix(tx, ty)).times(TransformUtilities.scaleMatrix(sx, sy)).times(TransformUtilities.translateMatrix(-tx, -ty));
    }

    public static Matrix scaleMatrixAboutPoint(double sx, double sy, Point2d point) {
        return Matrix.identity(3, 3).times(TransformUtilities.translateMatrix(point.getX(), point.getY())).times(TransformUtilities.scaleMatrix(sx, sy)).times(TransformUtilities.translateMatrix(-point.getX(), -point.getY()));
    }

    public static Matrix rotationMatrixAboutPoint(double rot, float tx, float ty) {
        return Matrix.identity(3, 3).times(TransformUtilities.translateMatrix(tx, ty)).times(TransformUtilities.rotationMatrix(rot)).times(TransformUtilities.translateMatrix(-tx, -ty));
    }

    @Reference(author={"Sp\"ath, Helmuth"}, title="Fitting affine and orthogonal transformations between two sets of points.", type=ReferenceType.Article, year="2004", journal="Mathematical Communications", publisher="Croatian Mathematical Society, Division Osijek, Osijek; Faculty of Electrical Engineering, University of Osijek, Osijek", pages={"27", "34"}, volume="9", number="1")
    public static Matrix affineMatrixND(double[][] q, double[][] p) {
        int dim = q[0].length;
        double[][] c = new double[dim + 1][dim];
        for (int j = 0; j < dim; ++j) {
            for (int k = 0; k < dim + 1; ++k) {
                for (int i = 0; i < q.length; ++i) {
                    double qtk = 1.0;
                    if (k < 3) {
                        qtk = q[i][k];
                    }
                    double[] dArray = c[k];
                    int n = j;
                    dArray[n] = dArray[n] + qtk * p[i][j];
                }
            }
        }
        double[][] Q = new double[dim + 1][dim + 1];
        for (double[] qt : q) {
            for (int i = 0; i < dim + 1; ++i) {
                int j = 0;
                while (j < dim + 1) {
                    double qti = 1.0;
                    if (i < 3) {
                        qti = qt[i];
                    }
                    double qtj = 1.0;
                    if (j < 3) {
                        qtj = qt[j];
                    }
                    double[] dArray = Q[i];
                    int n = j++;
                    dArray[n] = dArray[n] + qti * qtj;
                }
            }
        }
        Matrix Qm = new Matrix(Q);
        Matrix cm = new Matrix(c);
        Matrix a = Qm.solve(cm);
        Matrix t = Matrix.identity(dim + 1, dim + 1);
        t.setMatrix(0, dim - 1, 0, dim, a.transpose());
        return t;
    }

    @Reference(author={"Sp\"ath, Helmuth"}, title="Fitting affine and orthogonal transformations between two sets of points.", type=ReferenceType.Article, year="2004", journal="Mathematical Communications", publisher="Croatian Mathematical Society, Division Osijek, Osijek; Faculty of Electrical Engineering, University of Osijek, Osijek", pages={"27", "34"}, volume="9", number="1")
    public static Matrix affineMatrixND(List<? extends IndependentPair<? extends Coordinate, ? extends Coordinate>> data) {
        int k;
        int dim = data.get(0).firstObject().getDimensions();
        int nItems = data.size();
        double[][] c = new double[dim + 1][dim];
        for (int j = 0; j < dim; ++j) {
            for (k = 0; k < dim + 1; ++k) {
                for (int i = 0; i < nItems; ++i) {
                    double qtk = 1.0;
                    if (k < 3) {
                        qtk = data.get(i).firstObject().getOrdinate(k).doubleValue();
                    }
                    double[] dArray = c[k];
                    int n = j;
                    dArray[n] = dArray[n] + qtk * data.get(i).secondObject().getOrdinate(j).doubleValue();
                }
            }
        }
        double[][] Q = new double[dim + 1][dim + 1];
        for (k = 0; k < nItems; ++k) {
            int i;
            double[] qt = new double[dim + 1];
            for (i = 0; i < dim + 1; ++i) {
                qt[i] = data.get(k).firstObject().getOrdinate(i).doubleValue();
            }
            qt[dim] = 1.0;
            for (i = 0; i < dim + 1; ++i) {
                int j = 0;
                while (j < dim + 1) {
                    double qti = qt[i];
                    double qtj = qt[j];
                    double[] dArray = Q[i];
                    int n = j++;
                    dArray[n] = dArray[n] + qti * qtj;
                }
            }
        }
        Matrix Qm = new Matrix(Q);
        Matrix cm = new Matrix(c);
        Matrix a = Qm.solve(cm);
        Matrix t = Matrix.identity(dim + 1, dim + 1);
        t.setMatrix(0, dim - 1, 0, dim, a.transpose());
        return t;
    }

    @Reference(type=ReferenceType.Article, author={"Berthold K. P. Horn", "H.M. Hilden", "Shariar Negahdaripour"}, title="Closed-Form Solution of Absolute Orientation using Orthonormal Matrices", year="1988", journal="JOURNAL OF THE OPTICAL SOCIETY AMERICA", pages={"1127", "1135"}, number="7", volume="5")
    public static Matrix rigidMatrix(double[][] q, double[][] p) {
        int dim = q[0].length;
        int nitems = q.length;
        double[] qmean = new double[dim];
        double[] pmean = new double[dim];
        for (int j = 0; j < nitems; ++j) {
            for (int i = 0; i < dim; ++i) {
                int n = i;
                qmean[n] = qmean[n] + q[j][i];
                int n2 = i;
                pmean[n2] = pmean[n2] + p[j][i];
            }
        }
        int i = 0;
        while (i < dim) {
            int n = i;
            qmean[n] = qmean[n] / (double)nitems;
            int n3 = i++;
            pmean[n3] = pmean[n3] / (double)nitems;
        }
        double[][] M = new double[dim][dim];
        for (int k = 0; k < nitems; ++k) {
            for (int j = 0; j < dim; ++j) {
                for (int i2 = 0; i2 < dim; ++i2) {
                    double[] dArray = M[j];
                    int n = i2;
                    dArray[n] = dArray[n] + (p[k][j] - pmean[j]) * (q[k][i2] - qmean[i2]);
                }
            }
        }
        Matrix Mm = new Matrix(M);
        Matrix Qm = Mm.transpose().times(Mm);
        Matrix QmInvSqrt = MatrixUtils.invSqrtSym(Qm);
        Matrix R = Mm.times(QmInvSqrt);
        Matrix pm = new Matrix(new double[][]{pmean}).transpose();
        Matrix qm = new Matrix(new double[][]{qmean}).transpose();
        Matrix T = pm.minus(R.times(qm));
        Matrix tf = Matrix.identity(dim + 1, dim + 1);
        tf.setMatrix(0, dim - 1, 0, dim - 1, R);
        tf.setMatrix(0, dim - 1, dim, dim, T);
        return tf;
    }

    @Reference(type=ReferenceType.Article, author={"Berthold K. P. Horn", "H.M. Hilden", "Shariar Negahdaripour"}, title="Closed-Form Solution of Absolute Orientation using Orthonormal Matrices", year="1988", journal="JOURNAL OF THE OPTICAL SOCIETY AMERICA", pages={"1127", "1135"}, number="7", volume="5")
    public static Matrix rigidMatrix(List<? extends IndependentPair<? extends Coordinate, ? extends Coordinate>> data) {
        int dim = data.get(0).firstObject().getDimensions();
        int nitems = data.size();
        double[] qmean = new double[dim];
        double[] pmean = new double[dim];
        for (int j = 0; j < nitems; ++j) {
            for (int i = 0; i < dim; ++i) {
                int n = i;
                qmean[n] = qmean[n] + data.get(j).firstObject().getOrdinate(i).doubleValue();
                int n2 = i;
                pmean[n2] = pmean[n2] + data.get(j).secondObject().getOrdinate(i).doubleValue();
            }
        }
        int i = 0;
        while (i < dim) {
            int n = i;
            qmean[n] = qmean[n] / (double)nitems;
            int n3 = i++;
            pmean[n3] = pmean[n3] / (double)nitems;
        }
        double[][] M = new double[dim][dim];
        for (int k = 0; k < nitems; ++k) {
            for (int j = 0; j < dim; ++j) {
                for (int i2 = 0; i2 < dim; ++i2) {
                    double[] dArray = M[j];
                    int n = i2;
                    dArray[n] = dArray[n] + (data.get(k).secondObject().getOrdinate(j).doubleValue() - pmean[j]) * (data.get(k).firstObject().getOrdinate(i2).doubleValue() - qmean[i2]);
                }
            }
        }
        Matrix Mm = new Matrix(M);
        Matrix Qm = Mm.transpose().times(Mm);
        Matrix QmInvSqrt = MatrixUtils.invSqrtSym(Qm);
        Matrix R = Mm.times(QmInvSqrt);
        Matrix pm = new Matrix(new double[][]{pmean}).transpose();
        Matrix qm = new Matrix(new double[][]{qmean}).transpose();
        Matrix T = pm.minus(R.times(qm));
        Matrix tf = Matrix.identity(dim + 1, dim + 1);
        tf.setMatrix(0, dim - 1, 0, dim - 1, R);
        tf.setMatrix(0, dim - 1, dim, dim, T);
        return tf;
    }

    public static Matrix affineMatrix(List<? extends IndependentPair<? extends Point2d, ? extends Point2d>> data) {
        Matrix transform = new Matrix(3, 3);
        transform.set(2, 0, 0.0);
        transform.set(2, 1, 0.0);
        transform.set(2, 2, 1.0);
        Matrix A = new Matrix(data.size() * 2, 7);
        int i = 0;
        int j = 0;
        while (i < data.size()) {
            float x1 = data.get(i).firstObject().getX();
            float y1 = data.get(i).firstObject().getY();
            float x2 = data.get(i).secondObject().getX();
            float y2 = data.get(i).secondObject().getY();
            A.set(j, 0, x1);
            A.set(j, 1, y1);
            A.set(j, 2, 1.0);
            A.set(j, 3, 0.0);
            A.set(j, 4, 0.0);
            A.set(j, 5, 0.0);
            A.set(j, 6, -x2);
            A.set(j + 1, 0, 0.0);
            A.set(j + 1, 1, 0.0);
            A.set(j + 1, 2, 0.0);
            A.set(j + 1, 3, x1);
            A.set(j + 1, 4, y1);
            A.set(j + 1, 5, 1.0);
            A.set(j + 1, 6, -y2);
            ++i;
            j += 2;
        }
        double[] W = MatrixUtils.solveHomogeneousSystem(A);
        transform.set(0, 0, W[0] / W[6]);
        transform.set(0, 1, W[1] / W[6]);
        transform.set(0, 2, W[2] / W[6]);
        transform.set(1, 0, W[3] / W[6]);
        transform.set(1, 1, W[4] / W[6]);
        transform.set(1, 2, W[5] / W[6]);
        return transform;
    }

    public static Matrix scaleMatrix(double d, double e) {
        Matrix scaleMatrix = new Matrix(new double[][]{{d, 0.0, 0.0}, {0.0, e, 0.0}, {0.0, 0.0, 1.0}});
        return scaleMatrix;
    }

    public static Pair<Matrix> getNormalisations(List<? extends IndependentPair<? extends Point2d, ? extends Point2d>> data) {
        Point2dImpl firstMean = new Point2dImpl(0.0f, 0.0f);
        Point2dImpl secondMean = new Point2dImpl(0.0f, 0.0f);
        for (IndependentPair<? extends Point2d, ? extends Point2d> independentPair : data) {
            firstMean.x += independentPair.firstObject().getX();
            firstMean.y += independentPair.firstObject().getY();
            secondMean.x += independentPair.secondObject().getX();
            secondMean.y += independentPair.secondObject().getY();
        }
        firstMean.x /= (float)data.size();
        firstMean.y /= (float)data.size();
        secondMean.x /= (float)data.size();
        secondMean.y /= (float)data.size();
        Point2dImpl firstStd = new Point2dImpl(0.0f, 0.0f);
        Point2dImpl point2dImpl = new Point2dImpl(0.0f, 0.0f);
        for (IndependentPair<? extends Point2d, ? extends Point2d> independentPair : data) {
            firstStd.x = (float)((double)firstStd.x + Math.pow(firstMean.x - independentPair.firstObject().getX(), 2.0));
            firstStd.y = (float)((double)firstStd.y + Math.pow(firstMean.y - independentPair.firstObject().getY(), 2.0));
            point2dImpl.x = (float)((double)point2dImpl.x + Math.pow(secondMean.x - independentPair.secondObject().getX(), 2.0));
            point2dImpl.y = (float)((double)point2dImpl.y + Math.pow(secondMean.y - independentPair.secondObject().getY(), 2.0));
        }
        firstStd.x = (double)firstStd.x < 1.0E-5 ? 1.0f : firstStd.x;
        firstStd.y = (double)firstStd.y < 1.0E-5 ? 1.0f : firstStd.y;
        point2dImpl.x = (double)point2dImpl.x < 1.0E-5 ? 1.0f : point2dImpl.x;
        point2dImpl.y = (double)point2dImpl.y < 1.0E-5 ? 1.0f : point2dImpl.y;
        firstStd.x = (float)(Math.sqrt(2.0) / Math.sqrt(firstStd.x / (float)(data.size() - 1)));
        firstStd.y = (float)(Math.sqrt(2.0) / Math.sqrt(firstStd.y / (float)(data.size() - 1)));
        point2dImpl.x = (float)(Math.sqrt(2.0) / Math.sqrt(point2dImpl.x / (float)(data.size() - 1)));
        point2dImpl.y = (float)(Math.sqrt(2.0) / Math.sqrt(point2dImpl.y / (float)(data.size() - 1)));
        Matrix firstMatrix = new Matrix(new double[][]{{firstStd.x, 0.0, -firstMean.x * firstStd.x}, {0.0, firstStd.y, -firstMean.y * firstStd.y}, {0.0, 0.0, 1.0}});
        Matrix matrix = new Matrix(new double[][]{{point2dImpl.x, 0.0, -secondMean.x * point2dImpl.x}, {0.0, point2dImpl.y, -secondMean.y * point2dImpl.y}, {0.0, 0.0, 1.0}});
        return new Pair<Matrix>(firstMatrix, matrix);
    }

    public static List<? extends IndependentPair<Point2d, Point2d>> normalise(List<? extends IndependentPair<Point2d, Point2d>> data, Pair<Matrix> normalisations) {
        ArrayList<Pair<Point2d>> normData = new ArrayList<Pair<Point2d>>();
        for (int i = 0; i < data.size(); ++i) {
            Point2d p1 = data.get(i).firstObject().transform((Matrix)normalisations.firstObject());
            Point2d p2 = data.get(i).secondObject().transform((Matrix)normalisations.secondObject());
            normData.add(new Pair<Point2d>(p1, p2));
        }
        return normData;
    }

    public static IndependentPair<Point2d, Point2d> normalise(IndependentPair<Point2d, Point2d> data, Pair<Matrix> normalisations) {
        Point2d p1 = data.firstObject().transform((Matrix)normalisations.firstObject());
        Point2d p2 = data.secondObject().transform((Matrix)normalisations.secondObject());
        return new Pair<Point2d>(p1, p2);
    }

    public static Matrix fundamentalMatrix8PtNorm(List<? extends IndependentPair<Point2d, Point2d>> data) {
        Pair<Matrix> normalisations = TransformUtilities.getNormalisations(data);
        Matrix A = new Matrix(data.size(), 9);
        for (int i = 0; i < data.size(); ++i) {
            Point2d p1 = data.get(i).firstObject().transform((Matrix)normalisations.firstObject());
            Point2d p2 = data.get(i).secondObject().transform((Matrix)normalisations.secondObject());
            float x1_1 = p1.getX();
            float x1_2 = p1.getY();
            float x2_1 = p2.getX();
            float x2_2 = p2.getY();
            A.set(i, 0, x2_1 * x1_1);
            A.set(i, 1, x2_1 * x1_2);
            A.set(i, 2, x2_1);
            A.set(i, 3, x2_2 * x1_1);
            A.set(i, 4, x2_2 * x1_2);
            A.set(i, 5, x2_2);
            A.set(i, 6, x1_1);
            A.set(i, 7, x1_2);
            A.set(i, 8, 1.0);
        }
        double[] W = MatrixUtils.solveHomogeneousSystem(A);
        Matrix fundamental = new Matrix(3, 3);
        fundamental.set(0, 0, W[0]);
        fundamental.set(0, 1, W[1]);
        fundamental.set(0, 2, W[2]);
        fundamental.set(1, 0, W[3]);
        fundamental.set(1, 1, W[4]);
        fundamental.set(1, 2, W[5]);
        fundamental.set(2, 0, W[6]);
        fundamental.set(2, 1, W[7]);
        fundamental.set(2, 2, W[8]);
        fundamental = MatrixUtils.reduceRank(fundamental, 2);
        fundamental = ((Matrix)normalisations.secondObject()).transpose().times(fundamental).times((Matrix)normalisations.firstObject());
        return fundamental;
    }

    public static Matrix fundamentalMatrix8Pt(List<? extends IndependentPair<Point2d, Point2d>> data) {
        Matrix A = new Matrix(data.size(), 9);
        for (int i = 0; i < data.size(); ++i) {
            Point2d p1 = data.get(i).firstObject();
            Point2d p2 = data.get(i).secondObject();
            float x1_1 = p1.getX();
            float x1_2 = p1.getY();
            float x2_1 = p2.getX();
            float x2_2 = p2.getY();
            A.set(i, 0, x2_1 * x1_1);
            A.set(i, 1, x2_1 * x1_2);
            A.set(i, 2, x2_1);
            A.set(i, 3, x2_2 * x1_1);
            A.set(i, 4, x2_2 * x1_2);
            A.set(i, 5, x2_2);
            A.set(i, 6, x1_1);
            A.set(i, 7, x1_2);
            A.set(i, 8, 1.0);
        }
        double[] W = MatrixUtils.solveHomogeneousSystem(A);
        Matrix fundamental = new Matrix(3, 3);
        fundamental.set(0, 0, W[0]);
        fundamental.set(0, 1, W[1]);
        fundamental.set(0, 2, W[2]);
        fundamental.set(1, 0, W[3]);
        fundamental.set(1, 1, W[4]);
        fundamental.set(1, 2, W[5]);
        fundamental.set(2, 0, W[6]);
        fundamental.set(2, 1, W[7]);
        fundamental.set(2, 2, W[8]);
        fundamental = MatrixUtils.reduceRank(fundamental, 2);
        return fundamental;
    }

    public static Matrix homographyMatrixNorm(List<? extends IndependentPair<? extends Point2d, ? extends Point2d>> data) {
        Pair<Matrix> normalisations = TransformUtilities.getNormalisations(data);
        Matrix A = new Matrix(data.size() * 2, 9);
        int i = 0;
        int j = 0;
        while (i < data.size()) {
            Point2d p1 = data.get(i).firstObject().transform((Matrix)normalisations.firstObject());
            Point2d p2 = data.get(i).secondObject().transform((Matrix)normalisations.secondObject());
            float x1 = p1.getX();
            float y1 = p1.getY();
            float x2 = p2.getX();
            float y2 = p2.getY();
            A.set(j, 0, x1);
            A.set(j, 1, y1);
            A.set(j, 2, 1.0);
            A.set(j, 3, 0.0);
            A.set(j, 4, 0.0);
            A.set(j, 5, 0.0);
            A.set(j, 6, -(x2 * x1));
            A.set(j, 7, -(x2 * y1));
            A.set(j, 8, -x2);
            A.set(j + 1, 0, 0.0);
            A.set(j + 1, 1, 0.0);
            A.set(j + 1, 2, 0.0);
            A.set(j + 1, 3, x1);
            A.set(j + 1, 4, y1);
            A.set(j + 1, 5, 1.0);
            A.set(j + 1, 6, -(y2 * x1));
            A.set(j + 1, 7, -(y2 * y1));
            A.set(j + 1, 8, -y2);
            ++i;
            j += 2;
        }
        double[] W = MatrixUtils.solveHomogeneousSystem(A);
        Matrix homography = new Matrix(3, 3);
        homography.set(0, 0, W[0]);
        homography.set(0, 1, W[1]);
        homography.set(0, 2, W[2]);
        homography.set(1, 0, W[3]);
        homography.set(1, 1, W[4]);
        homography.set(1, 2, W[5]);
        homography.set(2, 0, W[6]);
        homography.set(2, 1, W[7]);
        homography.set(2, 2, W[8]);
        homography = ((Matrix)normalisations.secondObject()).inverse().times(homography).times((Matrix)normalisations.firstObject());
        if (Math.abs(homography.get(2, 2)) > 1.0E-6) {
            MatrixUtils.times(homography, 1.0 / homography.get(2, 2));
        }
        return homography;
    }

    public static Matrix homographyMatrix(List<? extends IndependentPair<? extends Point2d, ? extends Point2d>> data) {
        Matrix A = new Matrix(data.size() * 2, 9);
        int i = 0;
        int j = 0;
        while (i < data.size()) {
            Point2d p1 = data.get(i).firstObject();
            Point2d p2 = data.get(i).secondObject();
            float x1 = p1.getX();
            float y1 = p1.getY();
            float x2 = p2.getX();
            float y2 = p2.getY();
            A.set(j, 0, x1);
            A.set(j, 1, y1);
            A.set(j, 2, 1.0);
            A.set(j, 3, 0.0);
            A.set(j, 4, 0.0);
            A.set(j, 5, 0.0);
            A.set(j, 6, -(x2 * x1));
            A.set(j, 7, -(x2 * y1));
            A.set(j, 8, -x2);
            A.set(j + 1, 0, 0.0);
            A.set(j + 1, 1, 0.0);
            A.set(j + 1, 2, 0.0);
            A.set(j + 1, 3, x1);
            A.set(j + 1, 4, y1);
            A.set(j + 1, 5, 1.0);
            A.set(j + 1, 6, -(y2 * x1));
            A.set(j + 1, 7, -(y2 * y1));
            A.set(j + 1, 8, -y2);
            ++i;
            j += 2;
        }
        double[] W = MatrixUtils.solveHomogeneousSystem(A);
        Matrix homography = new Matrix(3, 3);
        homography.set(0, 0, W[0]);
        homography.set(0, 1, W[1]);
        homography.set(0, 2, W[2]);
        homography.set(1, 0, W[3]);
        homography.set(1, 1, W[4]);
        homography.set(1, 2, W[5]);
        homography.set(2, 0, W[6]);
        homography.set(2, 1, W[7]);
        homography.set(2, 2, W[8]);
        if (Math.abs(homography.get(2, 2)) > 1.0E-6) {
            MatrixUtils.times(homography, 1.0 / homography.get(2, 2));
        }
        return homography;
    }

    public static Matrix homographyToAffine(Matrix homography) {
        double h11 = homography.get(0, 0);
        double h12 = homography.get(0, 1);
        double h13 = homography.get(0, 2);
        double h21 = homography.get(1, 0);
        double h22 = homography.get(1, 1);
        double h23 = homography.get(1, 2);
        double h31 = homography.get(2, 0);
        double h32 = homography.get(2, 1);
        double h33 = homography.get(2, 2);
        Matrix affine = new Matrix(3, 3);
        affine.set(0, 0, h11 - h13 * h31 / h33);
        affine.set(0, 1, h12 - h13 * h32 / h33);
        affine.set(0, 2, h13 / h33);
        affine.set(1, 0, h21 - h23 * h31 / h33);
        affine.set(1, 1, h22 - h23 * h32 / h33);
        affine.set(1, 2, h23 / h33);
        affine.set(2, 0, 0.0);
        affine.set(2, 1, 0.0);
        affine.set(2, 2, 1.0);
        return affine;
    }

    public static Matrix homographyToAffine(Matrix homography, double x, double y) {
        double h11 = homography.get(0, 0);
        double h12 = homography.get(0, 1);
        double h13 = homography.get(0, 2);
        double h21 = homography.get(1, 0);
        double h22 = homography.get(1, 1);
        double h23 = homography.get(1, 2);
        double h31 = homography.get(2, 0);
        double h32 = homography.get(2, 1);
        double h33 = homography.get(2, 2);
        Matrix affine = new Matrix(3, 3);
        double fxdx = h11 / (h31 * x + h32 * y + h33) - (h11 * x + h12 * y + h13) * h31 / Math.pow(h31 * x + h32 * y + h33, 2.0);
        double fxdy = h12 / (h31 * x + h32 * y + h33) - (h11 * x + h12 * y + h13) * h32 / Math.pow(h31 * x + h32 * y + h33, 2.0);
        double fydx = h21 / (h31 * x + h32 * y + h33) - (h21 * x + h22 * y + h23) * h31 / Math.pow(h31 * x + h32 * y + h33, 2.0);
        double fydy = h22 / (h31 * x + h32 * y + h33) - (h21 * x + h22 * y + h23) * h32 / Math.pow(h31 * x + h32 * y + h33, 2.0);
        affine.set(0, 0, fxdx);
        affine.set(0, 1, fxdy);
        affine.set(0, 2, 0.0);
        affine.set(1, 0, fydx);
        affine.set(1, 1, fydy);
        affine.set(1, 2, 0.0);
        affine.set(2, 0, 0.0);
        affine.set(2, 1, 0.0);
        affine.set(2, 2, 1.0);
        return affine;
    }

    public static Matrix makeTransform(Rectangle from, Rectangle to) {
        Point2d trans = to.getTopLeft().minus(from.getTopLeft());
        double scaleW = to.getWidth() / from.getWidth();
        double scaleH = to.getHeight() / from.getHeight();
        return TransformUtilities.scaleMatrix(scaleW, scaleH).times(TransformUtilities.translateMatrix(trans.getX(), trans.getY()));
    }

    public static Matrix approximateRotationMatrix(Matrix approx) {
        SingularValueDecomposition svd = approx.svd();
        return svd.getU().times(svd.getV().transpose());
    }

    public static double[] rodrigues(Matrix R) {
        double w_norm = Math.acos((R.trace() - 1.0) / 2.0);
        if (w_norm < 1.0E-9) {
            return new double[3];
        }
        double norm = w_norm / (2.0 * Math.sin(w_norm));
        return new double[]{norm * (R.get(2, 1) - R.get(1, 2)), norm * (R.get(0, 2) - R.get(2, 0)), norm * (R.get(1, 0) - R.get(0, 1))};
    }

    public static Matrix rodrigues(double[] r) {
        Matrix wx = new Matrix(new double[][]{{0.0, -r[2], r[1]}, {r[2], 0.0, -r[0]}, {-r[1], r[0], 0.0}});
        double norm = Math.sqrt(r[0] * r[0] + r[1] * r[1] + r[2] * r[2]);
        if (norm < 1.0E-10) {
            return Matrix.identity(3, 3);
        }
        Matrix t1 = wx.times(Math.sin(norm) / norm);
        Matrix t2 = wx.times(wx).times((1.0 - Math.cos(norm)) / (norm * norm));
        return Matrix.identity(3, 3).plus(t1).plus(t2);
    }
}

