function hybrid=myhybridimages(lowImage, lowSigma, highImage, highSigma)
	% MYHYBRIDIMAGES  Create hybrid images by combining a low-pass and high-pass filtered pair.
	%   C = MYHYBRIDIMAGES(lowImage, lowSigma, highImage, highSigma) returns the hybrid image created
	%       by low-pass filtering lowImage with a Gaussian of s.d. lowSigma and combining it with 
	%       a high-pass image created by subtracting highImage from highImage convolved with
	%       a Gaussian of s.d. highSigma
	%   
	%   lowImage and highImage should both have size (rows,cols) or (rows,cols,channels). 
	%   The resultant image also has the same size
	
    k1 = makegaussiankernel(lowSigma);
    k2 = makegaussiankernel(highSigma);
    lp = myconvolution(lowImage, k1);
    hp = highImage - myconvolution(highImage, k2);
    
	if size(size(lp))[1] == 3
		for c=1:3
			hp(:,:,c) = normalise(hp(:,:,c))
		end
	else
		hp = normalise(hp)
	end
	hybrid = lp+hp;
end

function f=makegaussiankernel(sigma)
	% Use this function to create a 2D gaussian kernel with standard deviation sigma.
	% The kernel values should sum to 1.0, and the size should be floor(8*sigma+1) or 
	% floor(8*sigma+1)+1 (whichever is odd) as per the assignment specification.
	
    sz = floor(8.0 * sigma + 1.0);
    if mod(sz,2) == 0
        sz = sz + 1;
    end
    
    f = zeros(sz, sz);
    hs = floor((sz - 1) / 2);
    sigmasq = sigma * sigma;

	j = 1;
    for y=-hs:hs
    		i = 1;
        for x=-hs:hs
            radsqrd = x * x + y * y;
            f(j, i) = exp(-radsqrd / (2 * sigmasq));
            i = i + 1;
        end
    		j = j + 1;
    end
    f = f / sum(sum(f));
end

function normalised=normalise(image)
%   Histogram normalisation to stretch from black to white
%   Usage: [new image]=normalise(image) 
%   Parameters: image-array of integers 
%   Author: Mark S. Nixon
%   get dimensions 
    [rows,cols]= size(image);
%   set minimum
    minim = min(min(image));
%   work out range of input levels
	range = max(max(image))-minim;
%   normalise the image
    for x=1:cols %address all columns
        for y=1:rows %address all rows 
            normalised(y,x)=floor((image(y,x)-minim)*255/range);
        end
    end
end


